/*
 * Decompiled with CFR 0.152.
 */
package de.murmelmeister.murmelapi.language.message;

import com.github.benmanes.caffeine.cache.LoadingCache;
import de.murmelmeister.library.database.Database;
import de.murmelmeister.murmelapi.language.message.Message;
import de.murmelmeister.murmelapi.utils.CacheUtil;
import de.murmelmeister.murmelapi.utils.ResultSetUtil;
import de.murmelmeister.murmelapi.utils.update.RefreshEvent;
import de.murmelmeister.murmelapi.utils.update.RefreshListener;
import de.murmelmeister.murmelapi.utils.update.RefreshType;
import de.murmelmeister.murmelapi.utils.update.RefreshUtil;
import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class MessageCache
implements RefreshListener,
AutoCloseable {
    private static final Pattern LANGUAGE_KEY_PATTERN = Pattern.compile("^LanguageKey\\[languageId=(\\d+)]$");
    private static final Pattern TAG_KEY_PATTERN = Pattern.compile("^TagKey\\[tagId=(\\w+), languageId=(\\d+)]$");
    private final Database database;
    private final String tableName;
    private final LoadingCache<Integer, Message> cacheById;
    private final LoadingCache<TagKey, Message> cacheByTag;
    private final LoadingCache<LanguageKey, List<Message>> cacheByLanguage;
    private final Long fetchLimit;

    public MessageCache(Database database, String tableName, Long fetchLimit, long cacheCapacity, Duration refreshInterval) {
        this.database = database;
        this.tableName = tableName;
        this.fetchLimit = fetchLimit;
        this.cacheById = CacheUtil.buildCacheRefresh(this::loadById, cacheCapacity, refreshInterval);
        this.cacheByTag = CacheUtil.buildCacheRefresh(this::loadByTag, cacheCapacity, refreshInterval);
        this.cacheByLanguage = CacheUtil.buildCacheRefresh(key -> this.loadByLanguage(key.languageId()), cacheCapacity, refreshInterval);
        RefreshUtil.register(this);
    }

    @Override
    public void onRefresh(RefreshEvent<?> event) {
        String cacheName = event.type();
        if (RefreshType.MESSAGES.getName().equalsIgnoreCase(cacheName) || RefreshType.ALL.getName().equalsIgnoreCase(cacheName)) {
            this.refreshAll();
        } else if (RefreshType.SINGLE_MESSAGE.getName().equalsIgnoreCase(cacheName)) {
            Object key = event.key();
            if (!(key instanceof String)) {
                if (key instanceof Integer) {
                    Integer id = (Integer)key;
                    this.refreshSingle(id);
                } else if (key instanceof TagKey) {
                    TagKey tagKey = (TagKey)key;
                    this.refreshSingle(tagKey);
                } else if (key instanceof LanguageKey) {
                    LanguageKey languageKey = (LanguageKey)key;
                    this.refreshSingle(languageKey);
                }
            } else {
                Matcher languageMatcher = LANGUAGE_KEY_PATTERN.matcher((String)key);
                Matcher tagMatcher = TAG_KEY_PATTERN.matcher((String)key);
                if (languageMatcher.matches()) {
                    int languageId = Integer.parseInt(languageMatcher.group(1));
                    this.refreshSingle(new LanguageKey(languageId));
                } else if (tagMatcher.matches()) {
                    String tagId = tagMatcher.group(1);
                    int languageId = Integer.parseInt(tagMatcher.group(2));
                    this.refreshSingle(new TagKey(tagId, languageId));
                } else {
                    int id = Integer.parseInt((String)key);
                    this.refreshSingle(id);
                }
            }
        }
    }

    @Override
    public void close() {
        RefreshUtil.unregister(this);
        this.clear();
    }

    private void refreshAll() {
        this.clear();
        List<Message> messages = this.loadAllFromDatabase();
        if (messages.isEmpty()) {
            return;
        }
        HashMap<LanguageKey, List> byLanguage = new HashMap<LanguageKey, List>();
        for (Message message : messages) {
            this.cacheById.put((Object)message.id(), (Object)message);
            this.cacheByTag.put((Object)new TagKey(message.tagId(), message.languageId()), (Object)message);
            LanguageKey languageKey = new LanguageKey(message.languageId());
            byLanguage.computeIfAbsent(languageKey, ignored -> new ArrayList()).add(message);
        }
        byLanguage.forEach((key, value) -> this.cacheByLanguage.put(key, List.copyOf(value)));
    }

    private void refreshSingle(LanguageKey key) {
        this.removeByLanguage(key.languageId());
        List<Message> messages = this.loadByLanguage(key.languageId());
        if (messages.isEmpty()) {
            return;
        }
        for (Message message : messages) {
            this.cacheById.put((Object)message.id(), (Object)message);
            this.cacheByTag.put((Object)new TagKey(message.tagId(), message.languageId()), (Object)message);
        }
        this.cacheByLanguage.put((Object)key, List.copyOf(messages));
    }

    private void refreshSingle(TagKey key) {
        this.removeByTag(key.tagId(), key.languageId());
        Message message = this.loadByTag(key);
        if (message != null) {
            this.put(message);
        }
    }

    private void refreshSingle(int id) {
        this.remove(id);
        Message message = this.loadById(id);
        if (message != null) {
            this.put(message);
        }
    }

    private List<Message> loadAllFromDatabase() {
        String sql = "SELECT * FROM " + this.tableName;
        return CacheUtil.loadList(this.database, sql, this.fetchLimit, ResultSetUtil.message());
    }

    private List<Message> loadByLanguage(int languageId) {
        String sql = "SELECT * FROM " + this.tableName + " WHERE language_id = ?";
        return CacheUtil.loadList(this.database, sql, this.fetchLimit, ResultSetUtil.message(), stmt -> stmt.setInt(1, languageId));
    }

    private Message loadByTag(TagKey key) {
        String sql = "SELECT * FROM " + this.tableName + " WHERE tag_id = ? AND language_id = ?";
        return CacheUtil.loadSingle(this.database, sql, this.fetchLimit, ResultSetUtil.message(), stmt -> {
            stmt.setString(1, key.tagId());
            stmt.setInt(2, key.languageId());
        });
    }

    private Message loadById(int id) {
        String sql = "SELECT * FROM " + this.tableName + " WHERE id = ?";
        return CacheUtil.loadSingle(this.database, sql, this.fetchLimit, ResultSetUtil.message(), stmt -> stmt.setInt(1, id));
    }

    public Message getById(int id) {
        return (Message)this.cacheById.get((Object)id);
    }

    public Message getByTag(String tagId, int languageId) {
        return (Message)this.cacheByTag.get((Object)new TagKey(tagId, languageId));
    }

    public List<Message> getByLanguage(int languageId) {
        return (List)this.cacheByLanguage.get((Object)new LanguageKey(languageId));
    }

    public void put(Message message) {
        this.cacheById.put((Object)message.id(), (Object)message);
        this.cacheByTag.put((Object)new TagKey(message.tagId(), message.languageId()), (Object)message);
        CacheUtil.put(this.cacheByLanguage, new LanguageKey(message.languageId()), message, v -> v.id() == message.id());
    }

    public void remove(int id) {
        Message removed = (Message)this.cacheById.getIfPresent((Object)id);
        if (removed != null) {
            this.cacheById.invalidate((Object)id);
            this.cacheByTag.invalidate((Object)new TagKey(removed.tagId(), removed.languageId()));
            CacheUtil.remove(this.cacheByLanguage, new LanguageKey(removed.languageId()), v -> v.id() == removed.id());
        }
    }

    public void removeByTag(String tag, int languageId) {
        TagKey key = new TagKey(tag, languageId);
        Message removed = (Message)this.cacheByTag.getIfPresent((Object)key);
        if (removed != null) {
            this.cacheByTag.invalidate((Object)key);
            this.cacheById.invalidate((Object)removed.id());
            CacheUtil.remove(this.cacheByLanguage, new LanguageKey(languageId), v -> v.id() == removed.id());
        }
    }

    public void removeByLanguage(int languageId) {
        LanguageKey langKey = new LanguageKey(languageId);
        List removed = (List)this.cacheByLanguage.getIfPresent((Object)langKey);
        if (removed != null) {
            removed.forEach(message -> {
                this.cacheById.invalidate((Object)message.id());
                this.cacheByTag.invalidate((Object)new TagKey(message.tagId(), message.languageId()));
            });
            this.cacheByLanguage.invalidate((Object)langKey);
        }
    }

    public void clear() {
        this.cacheById.invalidateAll();
        this.cacheByTag.invalidateAll();
        this.cacheByLanguage.invalidateAll();
    }

    protected record TagKey(String tagId, int languageId) {
    }

    protected record LanguageKey(int languageId) {
    }
}

